/* eslint-disable max-len */
import React, {useRef, useEffect, useCallback, useState} from 'react';
import {v4 as uuidv4} from 'uuid';
import RealTimeHuman from '@bddh/starling-realtime-client';

import './Demo.scss';

const Demo = () => {
    const [subTitle, setSubTitle] = useState('');
    const [audioMode, setAudioMode] = useState('btn');
    const [pullAudioFromRTC, setPullAudioFromRtc] = useState(false);
    const [isRecording, setIsRecording] = useState(false);
    const isRecordingRef = useRef(false);
    const [canTransfer, setCanTransfer] = useState(true);
    const canTransferRef = useRef(true);
    const enableInterruptRef = useRef(true);
    const humanInstanceRef = useRef(null);

    const handleDigitalHumanCallback = useCallback(async (data: any) => {
        const {status, content} = data;
        console.info(status, content);
        if (status === 'DH_LIB_MESSAGE') {
            switch (content.action) {
                case 'DOWN_SUBTITLE':
                    const {content: subtitleContent, completed, type} = JSON.parse(content.body);
                    if (type === 'QUERY') {
                        setSubTitle(subtitleContent); // 还在继续识别中completed：false
                        if (completed) {
                            console.info('完整询问内容：', subtitleContent);
                        }
                    }
                    if (type === 'REPLY') {
                        console.info('回答：', subtitleContent);
                    }
                    break;
                case 'AUDIO_QUERY_INTERRUPT':
                    console.info('停止发送音频');
                    break;
                case 'EMPTY_ASR_RESULT':
                    console.info('空ASR结果');
                    break;
                case 'ASR_ERROR':
                    console.info('ASR错误:', content);
                    break;
                default:
                    break;
            }
        }
    }, []);

    const handleConnect = useCallback(() => {
        setAudioMode('');
        humanInstanceRef.current = new RealTimeHuman({
            // token: '填写应用对应token',
            wrapperId: 'human-wrapper',
            connectParams: {
                // ttsPer: '填写音色id',
                // figureId: '填写人像id',
                resolutionHeight: 1920,
                resolutionWidth: 1080,
                inactiveDisconnectSec: 300
            },
            renderParams: {
                closeLog: true,
                autoChromaKey: false,
                fullStatus: true
            },
            onDigitalHumanCallback: handleDigitalHumanCallback
        });
        humanInstanceRef.current.createServer();
    }, [handleDigitalHumanCallback]);

    const handleConnectWithAudio = useCallback((mode = 'btn' as 'btn' | 'rtc') => {
        // 先销毁旧实例，避免重复创建导致 DOM ID 冲突
        if (humanInstanceRef.current) {
            humanInstanceRef.current = null;
        }
        setAudioMode(mode);
        setPullAudioFromRtc(() => {
            return mode === 'rtc';
        });
        humanInstanceRef.current = new RealTimeHuman({
            // token: '填写应用对应token',
            wrapperId: 'human-wrapper',
            connectParams: {
                // ttsPer: '填写音色id',
                // figureId: '填写人像id',
                resolutionHeight: 1920,
                resolutionWidth: 1080,
                inactiveDisconnectSec: 3000,
                pullAudioFromRtc: mode === 'rtc',
                pickAudioMode: mode === 'btn' ? 'pressButton' : 'free',
                enableInterrupt: enableInterruptRef.current,
                // asrDdc: false,
                // asrPunc: false,
                // asrItn: false,
                // asrSensitiveWord: false,
                // asrBoostingTable: '拾音,心情,实验'
                // asrVadPauseTime: 200,
            },
            renderParams: {
                closeLog: true,
                autoChromaKey: false,
                fullStatus: true
            },
            onDigitalHumanCallback: handleDigitalHumanCallback
        });
        humanInstanceRef.current.createServer();
    }, [handleDigitalHumanCallback]);

    const handleDestroy = useCallback(() => {
        setSubTitle('');
        isRecordingRef.current = false;
        setIsRecording(false);
        humanInstanceRef.current?.destroy();
    }, []);

    const handleInterrupt = useCallback(async () => {
        console.info('发送打断');
        await humanInstanceRef.current?.interrupt();
        console.info('打断生效');
    }, []);

    const handleTextRender = useCallback(() => {
        humanInstanceRef.current.textRender({
            body: '你好，这是我的开场白自我介绍，我是数字人。今天天气不错，打算去哪里玩吗？我可以给你推荐一些网红打卡点，需要的话请告诉我哦！',
            onCallbackMsg: (res: any) => console.log(res)
        });
    }, []);

    const handleStreamTextRender = useCallback(() => {
        const commandId = uuidv4();
        humanInstanceRef.current?.textStreamRender({
            body: JSON.stringify({
                first: true,
                last: false,
                text: '你好，这是我的开场白自我介绍，我是数字人'
            }),
            requestId: commandId,
            onCallbackMsg: (res: any) => console.log(res)
        });

        humanInstanceRef.current?.textStreamRender({
            body: JSON.stringify({
                first: false,
                last: false,
                text: '你好，这是我的开场白自我介绍，我是数字人'
            }),
            requestId: commandId,
            // 测试流式驱动多次填写onCallbackMsg是否兼容
            onCallbackMsg: (res: any) => console.log(res)
        });

        humanInstanceRef.current?.textStreamRender({
            body: JSON.stringify({
                first: false,
                last: false,
                text: '你好，这是我的开场白自我介绍，我是数字人'
            }),
            requestId: commandId
        });

        setTimeout(() => {
            humanInstanceRef.current?.textStreamRender({
                body: JSON.stringify({
                    first: false,
                    last: false,
                    text: '今天天气,今天天气,今天天气今天天气,今天天气,今天天气今天天气,今天天气,今天天气'
                }),
                requestId: commandId
            });

            humanInstanceRef.current?.textStreamRender({
                body: JSON.stringify({
                    first: false,
                    last: true,
                    text: '很好很好很好很好很好很好'
                }),
                requestId: commandId
            });
        }, 500);
    }, []);

    const fetchData = async (url: string): Promise<ArrayBuffer> => {
        const response = await fetch(url, {
            method: 'GET'
        });

        if (!response.ok) {
            throw new Error(`HTTP error! Status: ${response.status}`);
        }

        // 将响应体解析为 ArrayBuffer
        const arrayBuffer = await response.arrayBuffer();
        return arrayBuffer;
    };

    const arrayBufferToBase64 = (buffer: any) => {
        let binary = '';
        const bytes = new Uint8Array(buffer);
        const len = bytes.byteLength;
        for (let i = 0; i < len; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return window.btoa(binary);
    };

    const waitStop = async (time: number) => {
        return new Promise<void>(resolve => {
            setTimeout(() => resolve(), time || 100);
        });
    };

    const handleStreamAudioRender = useCallback(async () => {
        const buffer = await fetchData('https://meta-human-editor-prd.cdn.bcebos.com/open-api%2FaudioTest.pcm');
        const unitLen = 2048; // 模拟切段
        const requestId = uuidv4();
        const len = Math.ceil(buffer.byteLength / unitLen);
        for (let i = 0; i < len; ++i) {
            const arrayBuffer = buffer.slice(i * unitLen, (i + 1) * unitLen);
            const base64String = arrayBufferToBase64(arrayBuffer); // pcm转base64
            if (i % 10 === 0) {
                await waitStop(50);
            }
            humanInstanceRef.current?.audioStreamRender({
                requestId,
                body: JSON.stringify({
                    audio: base64String,
                    first: i === 0,
                    last: i === len - 1
                }),
                ...(i === 0 ? {onCallbackMsg: (res: any) => console.log(`audioRender ${JSON.stringify(res)}`)} : {})
            });
        }
    }, []);

    const handleMuteHuman = () => humanInstanceRef.current?.muteHuman();

    const handleUnMuteHuman = () => humanInstanceRef.current?.unMuteHuman();

    const handlePlayHuman = () => humanInstanceRef.current?.playHuman();

    const handlePauseHuman = () => humanInstanceRef.current?.pauseHuman();

    const changeMicrophoneState = useCallback(() => {
        humanInstanceRef.current?.muteMicrophone(canTransferRef.current);
        canTransferRef.current = !canTransferRef.current;
        setCanTransfer(canTransferRef.current);
    }, []);

    const startRecord = useCallback(async () => {
        if (isRecordingRef.current) {
            return;
        }
        // 当参数enableInterrupt为true时，主动发送打断消息可更快打断播报，不发送打断消息也可打断
        // 当参数enableInterrupt为false，请注释下面打断代码。
        if (enableInterruptRef.current) {
            await humanInstanceRef.current?.interrupt();
        }
        isRecordingRef.current = true;
        setIsRecording(true);
        humanInstanceRef.current?.startRecord();
    }, []);

    const stopRecord = useCallback(() => {
        if (!isRecordingRef.current) {
            return;
        }
        isRecordingRef.current = false;
        setIsRecording(false);
        humanInstanceRef.current?.stopRecord();
    }, []);

    useEffect(() => {
        if (audioMode === 'rtc') {
            humanInstanceRef.current?.startRecord();
        }
    }, [audioMode]);

    useEffect(() => {
        // 处理原生键盘事件（避免类型不匹配React的合成事件）
        const onKeyDown = (e: KeyboardEvent) => {
            e.preventDefault(); // 阻止空格触发页面滚动
            // 如果是长按触发的重复事件，直接返回
            if (e.repeat) {
                return;
            }
            if (audioMode === 'btn' && (e.code === 'Space' || e.keyCode === 32)) {
                startRecord();
            }
        };
        const onKeyUp = (e: KeyboardEvent) => {
            if (e.repeat) {
                return;
            }
            if (audioMode === 'btn' && (e.code === 'Space' || e.keyCode === 32)) {
                stopRecord();
            }
        };
        if (audioMode === 'btn') {
            document.addEventListener('keydown', onKeyDown);
            document.addEventListener('keyup', onKeyUp);
        }
        return () => {
            document.removeEventListener('keydown', onKeyDown);
            document.removeEventListener('keyup', onKeyUp);
        };
    }, [audioMode, startRecord, stopRecord]);

    // 组件卸载时清理实例
    useEffect(() => {
        return () => {
            if (humanInstanceRef.current) {
                humanInstanceRef.current = null;
            }
        };
    }, []);

    return (
        <div className="demo-wrapper">
            { audioMode &&
                <>
                    <h2>数字人拾音</h2>
                    <div style={{marginBottom: 16}}>
                        <div>当前拾音模式：{audioMode === 'rtc' ? '持续' : pullAudioFromRTC ? '按键拾音+rtc' : '按键拾音+ws'}</div>
                        {audioMode === 'btn' && (
                            <>
                                <button
                                    onTouchStart={e => {
                                        e.preventDefault();
                                        startRecord();
                                    }}
                                    onTouchEnd={e => {
                                        e.preventDefault();
                                        stopRecord();
                                    }}
                                    onContextMenu={e => e.preventDefault()} // 禁止长按弹出菜单（安卓/部分 WebView）
                                    onSelect={e => e.preventDefault()} // 有的浏览器会触发 selection
                                    style={{
                                        userSelect: 'none',
                                        WebkitUserSelect: 'none',
                                        WebkitTouchCallout: 'none', // iOS Safari 长按菜单
                                        touchAction: 'manipulation'
                                    }}
                                >{isRecording ? '正在拾音中............' : 'pc端按空格/移动端按此处说话'}
                                </button>
                            </>

                        )}
                        {audioMode === 'rtc' && (
                            <button onClick={() => changeMicrophoneState()}>{canTransfer ? '拾音中(点击静音麦克风)...' : '开启麦克风'}</button>
                        )}
                    </div>
                </>
            }
            <div className="human-container">
                <div id="human-wrapper"></div>
                <div className="sub-title">{subTitle}</div>
            </div>
            <h2>数字人创建与销毁</h2>
            <button onClick={handleConnect}>connect</button>
            <button onClick={() => handleConnectWithAudio()}>connect 按键拾音+ws</button>
            <button onClick={() => handleConnectWithAudio('rtc')}>connect 持续拾音+rtc</button>
            <button onClick={handleDestroy}>destroy</button>
            <h2>数字人驱动</h2>
            <button onClick={handleTextRender}>文本驱动</button>
            <button onClick={handleStreamTextRender}>流式文本驱动</button>
            <button onClick={handleStreamAudioRender}>流式音频驱动</button>
            <button onClick={handleInterrupt}>打断</button>
            <h2>数字人指令控制</h2>
            <button onClick={handleMuteHuman}>静音数字人</button>
            <button onClick={handleUnMuteHuman}>解除静音数字人</button>
            <button onClick={handlePlayHuman}>数字人播放</button>
            <button onClick={handlePauseHuman}>数字人暂停</button>
        </div>
    );
};

export default Demo;
