import React, {useRef, useEffect, useCallback} from 'react';
import {v4 as uuidv4} from 'uuid';
import RealTimeHuman from '@bddh/starling-realtime-client';

import './index.css';

const Demo = () => {
    const humanInstanceRef = useRef<any>(null);

    const handleDigitalHumanCallback = useCallback(async (data: any) => {
        const {status, content} = data;
        console.log(status, content);


    }, []);

    const handleConnect = useCallback(() => {
        humanInstanceRef.current = new RealTimeHuman({
            // token: '填写应用对应token',
            wrapperId: 'human-wrapper',
            connectParams: {
                // ttsPer: '填写音色id',
                // figureId: '填写人像id',
                resolutionHeight: 1920,
                resolutionWidth: 1080,
                inactiveDisconnectSec: 30,
            },
            renderParams: {
                closeLog: true,
                autoChromaKey: false,
                fullStatus: true,
            },
            onDigitalHumanCallback: handleDigitalHumanCallback
        });
        humanInstanceRef.current.createServer();
    }, [handleDigitalHumanCallback]);

    const handleDestory = useCallback(() => {
        humanInstanceRef.current?.destory();
    }, []);

    const handleInterrupt = useCallback(async () => {
        console.info('发送打断');
        await humanInstanceRef.current?.interrupt();
        console.info('打断生效');
    }, []);

    const handleTextRender = useCallback(() => {
        humanInstanceRef.current.textRender({
            body: '你好，这是我的开场白自我介绍，我是数字人',
            onCallbackMsg: (res: any) => console.log(res)
        });
    }, []);

    const handleStreamTextRender = useCallback(() => {
        const commandId = uuidv4();
        humanInstanceRef.current?.textStreamRender({
            body: JSON.stringify({
                first: true,
                last: false,
                text: '你好，这是我的开场白自我介绍，我是数字人'
            }),
            requestId: commandId,
            onCallbackMsg: (res: any) => console.log(res)
        });

        humanInstanceRef.current?.textStreamRender({
            body: JSON.stringify({
                first: false,
                last: false,
                text: '你好，这是我的开场白自我介绍，我是数字人'
            }),
            requestId: commandId,
            // 测试流式驱动多次填写onCallbackMsg是否兼容
            onCallbackMsg: (res: any) => console.log(res)
        });

        humanInstanceRef.current?.textStreamRender({
            body: JSON.stringify({
                first: false,
                last: false,
                text: '你好，这是我的开场白自我介绍，我是数字人'
            }),
            requestId: commandId
        });

        setTimeout(() => {
            humanInstanceRef.current?.textStreamRender({
                body: JSON.stringify({
                    first: false,
                    last: false,
                    text: '今天天气,今天天气,今天天气今天天气,今天天气,今天天气今天天气,今天天气,今天天气'
                }),
                requestId: commandId
            });

            humanInstanceRef.current?.textStreamRender({
                body: JSON.stringify({
                    first: false,
                    last: true,
                    text: '很好很好很好很好很好很好'
                }),
                requestId: commandId
            });
        }, 500);
    }, []);

    const fetchData = async (url: string): Promise<ArrayBuffer> => {
        const response = await fetch(url, {
            method: 'GET'
        });

        if (!response.ok) {
            throw new Error(`HTTP error! Status: ${response.status}`);
        }

        // 将响应体解析为 ArrayBuffer
        const arrayBuffer = await response.arrayBuffer();
        return arrayBuffer;
    };

    const arrayBufferToBase64 = (buffer: any) => {
        let binary = '';
        const bytes = new Uint8Array(buffer);
        const len = bytes.byteLength;
        for (let i = 0; i < len; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return window.btoa(binary);
    };

    const waitStop = async (time: number) => {
        return new Promise<void>(resolve => {
            setTimeout(() => resolve(), time || 100);
        });
    };

    const handleStreamAudioRender = useCallback(async () => {
        const buffer = await fetchData('https://meta-human-editor-prd.cdn.bcebos.com/open-api%2FaudioTest.pcm');
        const unitLen = 2048; // 模拟切段
        const requestId = uuidv4();
        const len = Math.ceil(buffer.byteLength / unitLen);
        for (let i = 0; i < len; ++i) {
            const arrayBuffer = buffer.slice(i * unitLen, (i + 1) * unitLen);
            const base64String = arrayBufferToBase64(arrayBuffer); // pcm转base64
            if (i % 10 === 0) {
                await waitStop(50);
            }
            humanInstanceRef.current?.audioStreamRender({
                requestId,
                body: JSON.stringify({
                    audio: base64String,
                    first: i === 0,
                    last: i === len - 1
                }),
                ...(i === 0 ? {onCallbackMsg: (res: any) => console.log(`audioRender ${JSON.stringify(res)}`)} : {})
            });
        }
    }, []);

    const handleMuteHuman = () => humanInstanceRef.current?.muteHuman();

    const handleUnMuteHuman = () => humanInstanceRef.current?.unMuteHuman();

    const handlePlayHuman = () => humanInstanceRef.current?.playHuman();

    const handlePauseHuman = () => humanInstanceRef.current?.pauseHuman();


    return (
        <div>
            <div style={{marginBottom: 16}}>
                <button onClick={() => humanInstanceRef.current?.startRecord()}>开启拾音</button>
                <button onClick={() => humanInstanceRef.current?.stopRecord()} style={{marginLeft: 8}}>关闭拾音</button>
            </div>
            <div style={{display: 'flex', overflow: 'auto'}}>
                <div style={{minWidth: 400, height: 800, backgroundColor: 'pink', position: 'relative'}}>
                    <div id="human-wrapper"></div>
                </div>
                <div style={{minWidth: 400, height: 800, backgroundColor: 'pink'}}>
                </div>
            </div>
            <h2>数字人创建与销毁</h2>
            <button onClick={handleConnect}>connect</button>
            <button onClick={handleDestory}>destory</button>
            <h2>数字人驱动</h2>
            <button onClick={handleTextRender}>文本驱动</button>
            <button onClick={handleStreamTextRender}>流式文本驱动</button>
            {/* <button onClick={handlePlayAudio}>音频驱动</button> */}
            <button onClick={handleStreamAudioRender}>流式音频驱动</button>
            <button onClick={handleInterrupt}>打断</button>
            <h2>数字人指令控制</h2>
            <button onClick={handleMuteHuman}>静音数字人</button>
            <button onClick={handleUnMuteHuman}>解除静音数字人</button>
            <button onClick={handlePlayHuman}>数字人播放</button>
            <button onClick={handlePauseHuman}>数字人暂停</button>
        </div>
    );
};

export default Demo;